%DOA estimation using sparse arrays and SS-MUSIC algorithm
%References
% [1] P. Pal and P. P. Vaidyanathan, Nested arrays: A novel approach to array processing with enhanced degrees of freedom, IEEE Trans. Signal Process., vol. 58, no. 8, pp. 4167C4181, Aug. 2010.
% [2] C. L. Liu and P. P.Vaidyanathan, Super nested arrays: Linear sparse arrays with reduced mutual coupling. Part I: Fundamentals, IEEE Trans. Signal Process., vol. 64, no. 15, pp. 3997-4012, Aug. 2016.
% [3] C. L. Liu and P. P. Vaidyanathan, Super nested arrays: Linear sparse arrays with reduced mutual coupling. Part II: High-order extensions,IEEE Trans. Signal Process., vol. 64, no. 16, pp. 4203-4217, Aug. 2016.
% [4] W. Shi, Y. Li, and S. A. Vorobyov, Low mutual coupling sparse array design using ULA fitting, in Proc. Int. Conf. Acoust., Speech Signal Process. (ICASSP), Toronto, Canada, Jun. 2021, pp. 2165C2168.
% [5] W. Shi, S. A. Vorobyov and Y. Li ULA Fitting for Sparse Array Design, IEEE Trans. Signal Process., vol. 69, no. 21, pp. 6431C6447, Nov. 2021.
clear(); close all;
SIZE = 16;
aratio = [2, 1, 1];
wavelength = 1; 
d = wavelength / 2;
n_source = 45;
theta_bar_max = 1/3;
doas = linspace(-theta_bar_max*pi, theta_bar_max*pi, n_source);
doas_nor = linspace(-theta_bar_max, theta_bar_max, n_source);
theta_bar = doas*(180/pi);
power_source = 1;
snapshot_count = 1000;
Npt = 361*5;
source_count = length(doas);
coupling_para = 0.3;
antenna_number = 30;
a = 9;
b = 13;

design_ana11 = ana11_1d(antenna_number, d);
design_ana12 = ana11_1d(antenna_number, d);     
design_nested = nested_1d(antenna_number, d);
design_supernested_2 = super_nested_1d(antenna_number, 2,d,'Super Nested Q = 2');
design_supernested_3 = super_nested_1d(antenna_number, 3,d,'Super Nested Q = 3');
design_MISC = MISC_1d(antenna_number, d);      
design_ATLI_1BL = ATLI_1BL_1d(antenna_number, d);       
design_ATLI_2BL = ATLI_2BL_1d(antenna_number, d);    
design_TSENA = tsena_1d(antenna_number, d);          
design_ATLI_2BL = ATLI_2BL_1d(antenna_number, d); 
design_UF_3BL = UF_3BLC1(antenna_number, d);      
design_UF_4BL = UF_3BLC1(antenna_number, d);
design_ePCA = ePCA_1d(a,b,d);
design_CP = coprime_1d(a,b,d);
       
designs = { ...
    design_MISC,...
    design_UF_3BL,...
    design_UF_4BL,...
    design_nested,...
    design_supernested_2,...
    design_supernested_3,...
    design_ana11,...
    design_ana12,...
    design_ATLI_1BL,...
    design_ATLI_2BL,...
    design_TSENA,...
    design_ePCA,...
    design_CP,...
};

designs_number = length(designs);
snr = 0;
power_noise = power_source/(10^((snr/10)));
n_repeat = 1;
spectrum_est_theta = zeros(designs_number,Npt);
spectrum_est_y = zeros(designs_number,Npt);
spectrum_est_target = zeros(designs_number,n_source);

 for rr = 1:n_repeat
        for dn = 1:designs_number
            design = designs{dn};
            [~, Coupling_Matrix,R, CL,~] = signal_considering_coupling(design, doas, wavelength, snapshot_count, power_noise, power_source,coupling_para);   
            [Rss, dss] = covariance_compute_1d(design, R);
            sp = rmusic_1d(Rss, source_count, 2*pi*design.element_spacing/wavelength,dss.element_count);
          
            spectrum_est_theta(dn,:) = spectrum_est_theta(dn,:) + (sp.theta);
            spectrum_est_y(dn,:) = spectrum_est_y(dn,:) + (sp.y);
            spectrum_est_target(dn,:) = spectrum_est_target(dn,:) + (sp.x);
        end
 end
 
spectrum_est_theta(:) = spectrum_est_theta(:)/ (n_repeat);
spectrum_est_y(:) = spectrum_est_y(:)/ (n_repeat);    
spectrum_est_target(:) = spectrum_est_target(:)/ (n_repeat);   

source_number = num2str(n_source);
sensor_number = num2str(antenna_number);
markers = {'+', 'o', 'd', '*','s','^','h','p','x','>', '+', 'o', 'd', '*','s','^','h','p','x','>','+', 'o', 'd', '*','s','^','h','p','x','>'};
colors  = {'k','b','r','g','m','c','k','b','r','g','m','c','k','b','r','g','m','c','k','b','r','g','m','c'};
lines = {'-',':','-.','--','-',':','-.','--','-',':','-.','--','-',':','-.','--'};

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%plot target
figure;
 ylabels = cell(1, designs_number+2);
for bb = 1:designs_number
       design = designs{bb};
       %SPmax(gg,:) = max(spectrum_est_y(gg,:));
       x = spectrum_est_target(bb,:)/pi*180;
       y = (designs_number+1-bb)*ones(1,n_source);
       h = plot(x, y,'kd');
%set(h,'Linewidth',2)

       xlabels = cell(1, n_source+2);
       xlabels{1} = '-90'; xlabels{n_source+2} = '90'; 
       xlabels{2} = num2str(-theta_bar_max*180);
       xlabels{n_source+1} = num2str(theta_bar_max*180);
       
       ylabels{1} = []; ylabels{n_source+2} = []; 
       ylabels{designs_number+2-bb} = designs{bb}.type; 
       
       
       grid on;
       
       axis([-90 90 0 designs_number+1]);
       
        set(gca,	'FontSize', SIZE, ...
        'XTick',  [-90; theta_bar'; 90], ...
        'XTickLabel', xlabels, ...
        'YTick', 0:1:designs_number+1,...
        'YTickLabel', ylabels);
       
       hold on;
       xlabel('angle (degree)')
      % ylabel('magnitude (dB)')   
       pbaspect(aratio)
      
       title([sensor_number, ' Sensors Detect ', source_number, ' Sources']);
end




